/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */       
/*
 *  HCI Command Driver on DM814X EVM.
 * 
 */

#include <stdio.h>
#include "uart.h"
#include "bluetooth.h"
#include "gpio.h"

#include "dm388_types.h"
#include "dm388_EVM.h"
#include "dm388_i2c.h"

#include "dm388_uart.h"

#define UART_TIMEOUT            (0x80000)
#define IOEXP_BT_EN_GPIO			(5)
#define MUX_SEL_GPIO			    (16)


/* Arbitary delay of a few usecs */
void delay(int delay)
{
	int i;
	
	for (i = 0; i < delay; i++);
}

int set_gpio_bank1 (int gpio, int value)
{
	unsigned int regval;
	static int initialised = 0;

	if (!initialised) {
	/* Reset GPIO Subsystem */
		GPIO1_SYSCONFIG = 0x00000020; /* Software Reset */
		delay(0x90000);
		GPIO1_SYSCONFIG = 0x100; /* no-idle */
		initialised = 1;
	}

	/* Output Enable GPIO16 in Bank 2 */
	regval = GPIO1_OE;
	regval = regval & ~(1 << gpio);
	GPIO1_OE = regval;

	if (!value) {
		/* Set WLAN_ENABLE Low */
		regval = GPIO1_DATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO1_DATAOUT = regval;
		regval = GPIO1_SETDATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO1_SETDATAOUT = regval;
	} else {
		/* Set BT_ENABLE High */
		GPIO1_DATAOUT |= (1 <<  gpio);
		GPIO1_SETDATAOUT |= (1 << gpio);
	}

	return 1;
}

int set_gpio_bank2 (int gpio, int value)
{
	unsigned int regval;
	static int initialised = 0;
		
	if (!initialised) {
	/* Reset GPIO Subsystem */
		GPIO2_SYSCONFIG = 0x00000020; /* Software Reset */
		delay(0x90000);
		GPIO2_SYSCONFIG = 0x100; /* no-idle */
		initialised = 1;
	}
	
	/* Output Enable GPIO16 in Bank 2 */
	regval = GPIO2_OE;
	regval = regval & ~(1 << gpio);  
	GPIO2_OE = regval;

	if (!value) {
		/* Set WLAN_ENABLE Low */
		regval = GPIO2_DATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO2_DATAOUT = regval;
		regval = GPIO2_SETDATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO2_SETDATAOUT = regval;
	} else {
		/* Set BT_ENABLE High */
		GPIO2_DATAOUT |= (1 <<  gpio);
		GPIO2_SETDATAOUT |= (1 << gpio);
	}
	
	return 1;
}

int dm388_evm_bt_init (void)
{

	/* Change MUX to support UART */
	/* A arbitary huge delay to make sure gpio is enabled */
	set_gpio_bank1 (MUX_SEL_GPIO, 1);
	delay(190000);
	set_gpio_bank1 (MUX_SEL_GPIO, 0);
	delay(190000);
	set_gpio_bank1 (MUX_SEL_GPIO, 1);
	delay(90000); /* settle down */

	/* Enable BT Chip */
	/* A arbitary huge delay to make sure BT chip is enabled */
	set_gpio_bank2 (IOEXP_BT_EN_GPIO, 1);
	delay(190000);
	set_gpio_bank2 (IOEXP_BT_EN_GPIO, 0);
	delay(190000);
	set_gpio_bank2 (IOEXP_BT_EN_GPIO, 1);
	delay(90000); /* settle down */
		
	return SUCCESS;
}

/* Send a HCI command over uart */
int 
send_hci_cmd (UART_Handle handle, unsigned char cmd, unsigned int opcode, unsigned char *data, int len)
{
        int err;

        if (!handle || (data == NULL && len > 0))
                return -1;

		err = uart_send (handle, HCI_CMD_PKT, UART_TIMEOUT);
        if (err)
                goto failed_to_send;
        err = uart_send (handle, opcode & 0xff, UART_TIMEOUT);
        if (err)
                goto failed_to_send;
                
        err = uart_send (handle, (opcode >> 8) & 0xff, UART_TIMEOUT);
        if (err)
                goto failed_to_send;

        err = uart_send (handle, len, UART_TIMEOUT);
        if (err)
                goto failed_to_send;

        while(len-- > 0 && data) {
                err = uart_send(handle, *data, UART_TIMEOUT);
                if (err)
                        goto failed_to_send;

                data++;
        }

        return 0;

failed_to_send:
        return 1;
}

INT32 get_btchip_version (UART_Handle handle, UINT32 * manfid)
{
	INT32 err;
	UINT8 resp[255];
	UINT8 plen;
	UINT8 opcode[2];
    UINT8 status;
	INT32 i;
	
	/* READ LOCAL VERSION of the BT Chip */
	err = send_hci_cmd(handle, HCI_CMD_PKT, HCI_READ_LOCAL_VER_CMD, NULL, 0);
	if (err)
		goto hci_cmd_send_err;
	
	err = uart_recv(handle, resp, UART_TIMEOUT);
	if (err)
		return err;

	if (resp[0] != HCI_EVENT_PKT)
		goto recvd_unknown_pkt;

	err = uart_recv(handle, resp, UART_TIMEOUT);
	if (err)
		goto recv_err;

 	if (resp[0] != COMMAND_COMPLETE_EVENT)
		goto err_cc_evnt;

	err = uart_recv(handle, &plen, UART_TIMEOUT);
	if (err)
		goto recv_err;

	err = uart_recv(handle, resp, UART_TIMEOUT);
	if (err)
		goto recv_err;

	if (resp[0] != HCI_CMD_PKT) /* Is event packet in response to CMD packet */
		goto recv_err;

	err = uart_recv(handle, &opcode[0], UART_TIMEOUT);
	if (err)
		goto recv_err;

	err = uart_recv(handle, &opcode[1], UART_TIMEOUT);
	if (err)
		goto recv_err;

#if 0
	/* Is event packet generated for RESET Command? */
	if (opcode != HCI_CMD_MANFID_OPCODE)
		goto recv_err;
#endif

	err = uart_recv(handle, &status, UART_TIMEOUT);
	if (err)
		goto recv_err;

	if (status != 0x0)
		goto recv_err;

	plen = plen - 4;
        
	for (i = 0; i < plen && i < 255; i++) {
		err = uart_recv(handle, &resp[i], UART_TIMEOUT);
		if (err)
        	goto recv_err;
	}
	
	*manfid = (resp[5] >> 8) | resp[4]; 	
		
	return SUCCESS;

err_cc_evnt:        
recvd_unknown_pkt:
hci_cmd_send_err:
recv_err:
        return FAILED;
}
